# SEO Audit Report Format Reference

**Version**: 1.0.0
**Last Updated**: October 2025

This document provides the structure and format for presenting WordLift SEO audit results to users.

**Important**: All reports MUST follow WordLift brand guidelines. See [resources/BRANDING.md](resources/BRANDING.md) for:
- Official color palette (Sky, Sand, Berry, Petal, Leaf, Moss)
- Typography standards (Open Sans font family)
- Logo usage and placement
- Design system patterns

---

## Report Structure

When presenting audit results, follow this hierarchical structure:

### 1. Executive Summary

**Required Elements**:
- **Overall SEO Health Score** (0-100) - Display with Sky-to-Moss gradient
- **Target URL** and domain name
- **Audit timestamp**
- **Quick verdict** with icon (e.g., "👍 Good Progress", "⚠️ Needs Improvement", "🔴 Critical Issues")
- **WordLift attribution** - "Powered by WordLift Enterprise SEO Platform"

**Color Coding for Scores**:
- 80-100: Leaf (`#22A286`) - Success
- 60-79: Sand (`#C2A41D`) - Warning
- 0-59: Berry (`#D55471`) - Critical

### 2. Quick Stats Dashboard

Present these key metrics prominently in a table with emoji icons:

- 📦 **Schema Types Found** - Count of unique schema types implemented
- 🔴 **Critical Issues** - Count of high-priority problems
- 💡 **Opportunities** - Count of growth opportunities identified
- ✅ **Passing Tests** - Count of successful validations

### 3. Score Breakdown

Provide detailed scores across these dimensions with color-coded progress bars:

| Dimension | Description | Weight |
|-----------|-------------|--------|
| **Schema Markup** | Structured data implementation quality | High |
| **Meta Tags** | Meta descriptions, titles, OG tags | Medium |
| **Content Quality** | Content richness and optimization | Medium |
| **Technical SEO** | Site structure, canonicals, redirects | High |
| **Mobile Ready** | Mobile optimization and responsive design | Medium |
| **Performance** | Page speed and Core Web Vitals | Medium |

Each score should be 0-100 and color-coded:
- **80-100**: Green (Excellent)
- **60-79**: Yellow (Good)
- **0-59**: Red (Needs Work)

### 4. Performance Radar Chart

Visualize the six dimensions above in a radar/spider chart to show:
- Your site's performance across all metrics
- Gaps and strengths at a glance
- Areas requiring immediate attention

### 5. Issues & Recommendations

List all identified issues with this structure:

```
[Severity Icon] [Priority Number] Issue Title [Impact Badge]

Description: Clear explanation of what's wrong and why it matters

Impact: HIGH/MEDIUM/LOW with specific consequences (e.g., "15-35% lower CTR")

Action Required:
- Specific implementation steps
- Schema.org types and properties to add
- Code examples where helpful
```

**Severity Levels:**
- 🔴 **Critical** - Major missing features affecting visibility
- ⚠️ **Warning** - Important improvements needed
- ℹ️ **Info** - Minor enhancements
- ✅ **Success** - Working correctly

**Impact Levels:**
- **HIGH** - Significant effect on rankings/traffic/conversions
- **MEDIUM** - Moderate improvement potential
- **LOW** - Minor optimization

### 6. Structured Data Inventory

List all schema types with status:

| Schema Type | Icon | Status | Count | Notes |
|-------------|------|--------|-------|-------|
| Product | 📦 | ✅ Found | 1 | Core schema present |
| Review | ⭐ | ❌ Missing | 0 | Add for rich snippets |
| AggregateRating | 📊 | ❌ Missing | 0 | Shows star ratings |
| Offer | 💰 | ✅ Found | 1 | Price data included |
| Brand | 🏷️ | ✅ Found | 1 | Brand identified |
| BreadcrumbList | 🗂️ | ✅ Found | 1 | Navigation breadcrumbs |
| Organization | 🏢 | ❌ Missing | 0 | For Knowledge Graph |
| FAQPage | ❓ | ❌ Missing | 0 | FAQ rich results |

### 7. Competitive Analysis

**Schema Comparison Chart:**

Show side-by-side comparison of schema types:
- Your site's implementation
- Average competitor implementation (top 3-5)
- Gaps where competitors are ahead

**Top Competitors Table:**

For each competitor in SERP results:

```
#[Rank] [Domain/URL]
Schema Score: [0-100]
Implemented Schemas: [Badge][Badge][Badge]...

Key advantages:
- What they're doing better
- Missing schemas you should add
```

### 8. Growth Opportunities

Prioritized list of actionable improvements:

```
[Number]. [Opportunity Title]
[HIGH/MEDIUM/LOW IMPACT] [LOW/MEDIUM/HIGH EFFORT]

Description: What this opportunity is and why it matters
Expected benefit: Specific gains (e.g., "Can increase CTR by 25%+")

Implementation Steps:
1. First action item
2. Second action item
3. Third action item

Resources: Links to schema.org docs, Google guidelines, etc.
```

## Data Structure Reference

When receiving data from WordLift MCP, expect this structure:

### Audit Response Schema

```json
{
  "score": 78,
  "targetUrl": "https://example.com/product",
  "timestamp": "2025-10-20T...",

  "scores": {
    "overall": 78,
    "schema": 65,
    "meta": 80,
    "content": 82,
    "technical": 85,
    "mobile": 90,
    "performance": 72
  },

  "structuredData": {
    "totalSchemas": 4,
    "types": [
      {
        "name": "Product",
        "count": 1,
        "status": "found",
        "properties": ["name", "description", "image", "offers"]
      }
    ]
  },

  "issues": [
    {
      "severity": "critical",
      "category": "Schema",
      "title": "Missing Review & Rating Schema",
      "description": "...",
      "impact": "HIGH",
      "actionable": "...",
      "priority": 1
    }
  ],

  "competitors": [
    {
      "rank": 1,
      "url": "competitor.com",
      "schemaScore": 92,
      "schemas": ["Product", "Review", "AggregateRating"]
    }
  ],

  "opportunities": [
    {
      "title": "Implement Review System",
      "impact": "HIGH",
      "effort": "MEDIUM",
      "description": "...",
      "steps": ["...", "..."]
    }
  ]
}
```

## Presentation Guidelines

### Use Rich Formatting

- **Bold** for emphasis on key terms
- `Code blocks` for schema properties and technical terms
- > Blockquotes for important callouts
- Tables for structured comparisons
- Lists for action items

### Color Coding (When Available)

- 🔴 Red: Critical issues, missing required schemas
- 🟡 Yellow: Warnings, medium priority items
- 🟢 Green: Successes, good implementations
- 🔵 Blue: Informational, context

### Icons for Quick Recognition

Use consistent emoji icons:
- 📊 Data/Analytics
- 🔴 Critical Issues
- ⚠️ Warnings
- ✅ Success/Passing
- 💡 Opportunities
- 📦 Product Schema
- ⭐ Reviews/Ratings
- 💰 Pricing/Offers
- 🏷️ Brand
- 🗂️ Navigation
- 🏢 Organization
- 📱 Mobile
- ⚡ Performance
- 🎯 Goals/Targets
- 🏆 Competition

### Actionable Language

Use clear, directive language:
- ✅ "Add Review schema with these properties: reviewRating, author, reviewBody"
- ✅ "Implement AggregateRating to show star ratings in search results"
- ❌ "You might want to consider possibly adding reviews"

### Prioritization

Always order issues by:
1. **Priority number** (1 = most important)
2. **Impact level** (HIGH > MEDIUM > LOW)
3. **Severity** (Critical > Warning > Info)

### Context and Benefits

For each recommendation, explain:
- **What** needs to be done
- **Why** it matters (business impact)
- **How** to implement (specific steps)
- **Expected results** (quantified when possible)

## Example Report Sections

### Executive Summary Example

```markdown
# 🎯 WordLift SEO Audit Results

**Target:** https://myblackbriar.com/collections/backpack/products/18l-extreme-adventure-backpack-7-0
**Date:** October 20, 2025
**Overall Score:** 78/100 👍 Good Progress

## Quick Stats
- 📊 **4** Schema Types Found
- 🔴 **2** Critical Issues
- 💡 **2** Growth Opportunities
- ✅ **2** Passing Tests
```

### Issues Example

```markdown
## 🎯 Critical Issues

### 🔴 Priority 1: Missing Review & Rating Schema [HIGH IMPACT]

Your product page lacks Review and AggregateRating structured data. Products with review markup can achieve **15-35% higher click-through rates** in search results.

**💡 Action Required:**

Implement these schema.org types:
- **Review** with properties: `reviewRating`, `author`, `reviewBody`, `datePublished`
- **AggregateRating** with properties: `ratingValue`, `reviewCount`, `bestRating`, `worstRating`

**Example Implementation:**
```json
{
  "@type": "Product",
  "aggregateRating": {
    "@type": "AggregateRating",
    "ratingValue": "4.5",
    "reviewCount": "89"
  }
}
```

**Resources:**
- [Google Review Snippets Guide](https://developers.google.com/search/docs/appearance/structured-data/review-snippet)
- [schema.org/Review](https://schema.org/Review)
```

### Opportunities Example

```markdown
## 💡 Growth Opportunities

### 1. Implement Review System [HIGH IMPACT] [MEDIUM EFFORT]

Add review schema to show star ratings in search results. Can increase CTR by 25%+ and build trust with potential customers.

**Implementation Steps:**
1. Collect customer reviews through your platform
2. Implement Review and AggregateRating schema markup
3. Test with [Google Rich Results Test](https://search.google.com/test/rich-results)
4. Monitor impressions and CTR in Search Console

**Expected Benefits:**
- ⭐ Star ratings in search results
- 📈 Higher click-through rates
- 🤝 Improved trust signals
- 🎯 Better visibility for product queries
```

## Interactive Elements (When Supported)

If the platform supports interactive visualizations:

- **Radar Charts** for multi-dimensional scoring
- **Bar Charts** for competitor comparisons
- **Progress Bars** for individual metric scores
- **Expandable Sections** for detailed technical data
- **Tabs** for organizing large reports

## Mobile Considerations

Ensure reports are readable on mobile:
- Responsive tables (stack on mobile)
- Touch-friendly expandable sections
- Readable font sizes (minimum 14px)
- Adequate spacing between elements

## Accessibility

- Use semantic heading hierarchy (h1 → h2 → h3)
- Provide alt text for icons/charts
- Ensure sufficient color contrast
- Support screen readers with proper markup

---

## Interactive Dashboard Reference Implementation

Below is a complete React component that implements all the formatting guidelines above. This can be used as a reference for building custom interactive dashboards.

### React Component: WordLiftSEODashboard.jsx

```jsx
import React, { useState } from 'react';
import { BarChart, Bar, XAxis, YAxis, CartesianGrid, Tooltip, Legend, ResponsiveContainer, RadarChart, PolarGrid, PolarAngleAxis, PolarRadiusAxis, Radar } from 'recharts';

const WordLiftSEODashboard = () => {
  const [targetUrl, setTargetUrl] = useState('https://myblackbriar.com/collections/backpack/products/18l-extreme-adventure-backpack-7-0');
  const [showInstructions, setShowInstructions] = useState(false);
  const [logoError, setLogoError] = useState(false);

  // WordLift Brand Colors
  const colors = {
    neutral900: '#191919',
    neutral500: '#A1A7AF',
    neutral100: '#F6F6F7',
    neutral0: '#FFFFFF',
    sky: '#3452DB',
    sand: '#C2A41D',
    berry: '#D55471',
    petal: '#A10269',
    leaf: '#22A286',
    moss: '#125054'
  };

  const BRAND_COLORS = [colors.sky, colors.sand, colors.berry, colors.petal, colors.leaf, colors.moss];

  // Full demonstration audit data
  const auditData = {
    score: 78,
    targetUrl: targetUrl,
    timestamp: new Date().toISOString(),
    structuredData: {
      totalSchemas: 4
    },
    scores: {
      overall: 78,
      schema: 65,
      meta: 80,
      content: 82,
      technical: 85,
      mobile: 90,
      performance: 72
    },
    issues: [
      {
        severity: 'critical',
        category: 'Schema',
        title: 'Missing Review & Rating Schema',
        description: 'Your product page lacks Review and AggregateRating structured data. Products with review markup can achieve 15-35% higher click-through rates.',
        impact: 'HIGH',
        actionable: 'Implement schema.org/Review and schema.org/AggregateRating with properties: reviewRating, author, reviewBody, ratingValue, reviewCount.',
        priority: 1
      },
      {
        severity: 'critical',
        category: 'Schema',
        title: 'Incomplete Product Schema Properties',
        description: 'Your Product schema is missing important identifiers: SKU, GTIN, MPN.',
        impact: 'MEDIUM',
        actionable: 'Add "sku", "gtin13" or "gtin14", and "mpn" properties to your Product schema.',
        priority: 2
      },
      {
        severity: 'warning',
        category: 'Schema',
        title: 'No Organization Schema',
        description: 'Missing Organization structured data reduces your brand\'s presence in Google\'s Knowledge Graph.',
        impact: 'MEDIUM',
        actionable: 'Add schema.org/Organization with properties: name, logo, url, sameAs, contactPoint.',
        priority: 3
      },
      {
        severity: 'success',
        category: 'Schema',
        title: '✓ Product Schema Implemented',
        description: 'Your page has valid Product structured data with essential properties.',
        priority: null
      },
      {
        severity: 'success',
        category: 'Meta',
        title: '✓ Meta Description Optimized',
        description: 'Your meta description is 157 characters - within the ideal range.',
        priority: null
      }
    ],
    schemaTypes: [
      { name: 'Product', count: 1, status: 'found', icon: '📦' },
      { name: 'Offer', count: 1, status: 'found', icon: '💰' },
      { name: 'Brand', count: 1, status: 'found', icon: '🏷️' },
      { name: 'Review', count: 0, status: 'missing', icon: '⭐' },
      { name: 'AggregateRating', count: 0, status: 'missing', icon: '📊' },
      { name: 'BreadcrumbList', count: 1, status: 'found', icon: '🗂️' },
      { name: 'Organization', count: 0, status: 'missing', icon: '🏢' }
    ],
    competitors: [
      { rank: 1, url: 'rei.com', schemaScore: 92, schemas: ['Product', 'Review', 'AggregateRating', 'Offer', 'Brand'] },
      { rank: 2, url: 'backcountry.com', schemaScore: 88, schemas: ['Product', 'Review', 'Offer', 'Brand'] },
      { rank: 3, url: 'patagonia.com', schemaScore: 95, schemas: ['Product', 'Review', 'AggregateRating', 'Offer', 'FAQPage'] }
    ],
    opportunities: [
      {
        title: 'Implement Review System',
        impact: 'HIGH',
        effort: 'MEDIUM',
        description: 'Add review schema to show star ratings. Can increase CTR by 25%+.',
        steps: ['Collect customer reviews', 'Implement Review schema', 'Test with Google Rich Results Test']
      },
      {
        title: 'Add Organization Schema',
        impact: 'MEDIUM',
        effort: 'LOW',
        description: 'Establish brand presence in Google Knowledge Graph.',
        steps: ['Add Organization schema', 'Include logo and social profiles', 'Add contact information']
      }
    ],
    radarData: [
      { metric: 'Schema', score: 65, fullMark: 100 },
      { metric: 'Meta Tags', score: 80, fullMark: 100 },
      { metric: 'Content', score: 82, fullMark: 100 },
      { metric: 'Technical', score: 85, fullMark: 100 },
      { metric: 'Mobile', score: 90, fullMark: 100 },
      { metric: 'Performance', score: 72, fullMark: 100 }
    ],
    schemaComparison: [
      { schema: 'Product', yourSite: 1, competitors: 3 },
      { schema: 'Review', yourSite: 0, competitors: 3 },
      { schema: 'Rating', yourSite: 0, competitors: 2 },
      { schema: 'Offer', yourSite: 1, competitors: 3 }
    ]
  };

  const getSeverityIcon = (severity) => {
    const icons = { critical: '🔴', warning: '⚠️', info: 'ℹ️', success: '✅' };
    return icons[severity] || '📋';
  };

  const getSeverityColor = (severity) => {
    const severityColors = {
      critical: colors.berry,
      warning: colors.sand,
      info: colors.sky,
      success: colors.leaf
    };
    return severityColors[severity] || colors.neutral500;
  };

  const getImpactBadge = (impact) => {
    const impactColors = { HIGH: colors.berry, MEDIUM: colors.sand, LOW: colors.sky };
    return (
      <span style={{
        background: impactColors[impact] || colors.neutral500,
        color: colors.neutral0,
        padding: '4px 12px',
        borderRadius: '12px',
        fontSize: '0.75em',
        fontWeight: '700',
        marginLeft: '8px',
        textTransform: 'uppercase'
      }}>
        {impact} IMPACT
      </span>
    );
  };

  const getEffortBadge = (effort) => {
    const effortColors = { LOW: colors.leaf, MEDIUM: colors.sand, HIGH: colors.berry };
    return (
      <span style={{
        background: effortColors[effort] || colors.neutral500,
        color: colors.neutral0,
        padding: '4px 12px',
        borderRadius: '12px',
        fontSize: '0.75em',
        fontWeight: '700',
        marginLeft: '8px',
        textTransform: 'uppercase'
      }}>
        {effort} EFFORT
      </span>
    );
  };

  // WordLift text logo as SVG for guaranteed display
  const WordLiftLogo = () => (
    <div style={{
      fontSize: '2.5em',
      fontWeight: '700',
      color: colors.neutral0,
      letterSpacing: '-1px',
      marginBottom: '10px',
      textAlign: 'center'
    }}>
      <span style={{ color: colors.neutral0 }}>Word</span>
      <span style={{ color: colors.sand }}>Lift</span>
    </div>
  );

  return (
    <div style={{
      minHeight: '100vh',
      background: `linear-gradient(135deg, ${colors.sky} 0%, ${colors.moss} 100%)`,
      padding: '20px',
      fontFamily: "'Open Sans', -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif"
    }}>
      <style>{`
        @import url('https://fonts.googleapis.com/css2?family=Open+Sans:wght@300;400;600;700&display=swap');
      `}</style>

      <div style={{ maxWidth: '1400px', margin: '0 auto' }}>
        {/* Header with Logo */}
        <div style={{
          textAlign: 'center',
          color: colors.neutral0,
          marginBottom: '30px',
          padding: '40px 30px'
        }}>
          {/* Try to load the image logo, fallback to text */}
          {!logoError ? (
            <img
              src="https://upload.wikimedia.org/wikipedia/commons/4/48/WordLift-logo-horizontal-2024.png"
              alt="WordLift Logo"
              onError={() => setLogoError(true)}
              style={{
                height: '50px',
                marginBottom: '20px',
                maxWidth: '100%',
                filter: 'brightness(0) invert(1)'
              }}
            />
          ) : (
            <WordLiftLogo />
          )}

          <h1 style={{
            fontSize: '2.5em',
            marginBottom: '15px',
            textShadow: '3px 3px 6px rgba(0,0,0,0.3)',
            fontWeight: '700',
            marginTop: '10px'
          }}>SEO Audit Dashboard</h1>
          <p style={{ fontSize: '1.1em', opacity: 0.95, fontWeight: '400' }}>
            Comprehensive Schema Analysis • Competitive Intelligence
          </p>
        </div>

        {/* Score Card */}
        <div style={{
          background: colors.neutral0,
          color: colors.neutral900,
          padding: '50px 40px',
          borderRadius: '12px',
          textAlign: 'center',
          marginBottom: '25px',
          boxShadow: '0 15px 50px rgba(0,0,0,0.2)',
          border: `4px solid ${colors.sky}`
        }}>
          <h2 style={{
            fontSize: '1.3em',
            marginBottom: '20px',
            fontWeight: '600',
            color: colors.moss,
            textTransform: 'uppercase',
            letterSpacing: '1px'
          }}>
            SEO Health Score
          </h2>
          <div style={{
            fontSize: '6em',
            fontWeight: '700',
            marginBottom: '15px',
            background: `linear-gradient(135deg, ${colors.sky} 0%, ${colors.moss} 100%)`,
            WebkitBackgroundClip: 'text',
            WebkitTextFillColor: 'transparent',
            backgroundClip: 'text'
          }}>
            {auditData.score}
          </div>
          <div style={{
            fontSize: '1.3em',
            padding: '10px 30px',
            background: colors.neutral100,
            borderRadius: '25px',
            display: 'inline-block',
            color: colors.neutral900,
            fontWeight: '600',
            border: `2px solid ${colors.sky}`
          }}>
            👍 Good Progress
          </div>
          <div style={{
            marginTop: '20px',
            fontSize: '0.9em',
            color: colors.neutral500,
            fontWeight: '600'
          }}>
            {new URL(auditData.targetUrl).hostname}
          </div>
        </div>

        {/* Quick Stats Grid */}
        <div style={{
          display: 'grid',
          gridTemplateColumns: 'repeat(auto-fit, minmax(220px, 1fr))',
          gap: '20px',
          marginBottom: '25px'
        }}>
          {[
            { label: 'Schema Types', value: auditData.structuredData.totalSchemas, icon: '📊', color: colors.sky },
            { label: 'Critical Issues', value: 2, icon: '🔴', color: colors.berry },
            { label: 'Opportunities', value: auditData.opportunities.length, icon: '💡', color: colors.sand },
            { label: 'Passing Tests', value: 2, icon: '✅', color: colors.leaf }
          ].map((metric, idx) => (
            <div key={idx} style={{
              background: colors.neutral0,
              padding: '25px',
              borderRadius: '12px',
              boxShadow: '0 8px 25px rgba(0,0,0,0.1)',
              borderLeft: `5px solid ${metric.color}`,
              textAlign: 'center',
              transition: 'all 0.3s ease',
              cursor: 'pointer'
            }}>
              <div style={{ fontSize: '2.5em', marginBottom: '10px' }}>{metric.icon}</div>
              <div style={{ fontSize: '3em', fontWeight: '700', color: metric.color, marginBottom: '5px' }}>
                {metric.value}
              </div>
              <div style={{ fontSize: '0.9em', color: colors.neutral500, textTransform: 'uppercase', fontWeight: '700' }}>
                {metric.label}
              </div>
            </div>
          ))}
        </div>

        {/* Score Breakdown */}
        <div style={{
          background: colors.neutral0,
          borderRadius: '12px',
          padding: '30px',
          marginBottom: '25px',
          boxShadow: '0 10px 40px rgba(0,0,0,0.1)',
          border: `3px solid ${colors.sky}`
        }}>
          <h2 style={{
            color: colors.moss,
            marginBottom: '25px',
            fontSize: '1.6em',
            fontWeight: '700',
            borderBottom: `3px solid ${colors.sky}`,
            paddingBottom: '10px'
          }}>
            📊 Score Breakdown
          </h2>
          <div style={{ display: 'grid', gap: '18px' }}>
            {[
              { label: 'Schema Markup', score: 65, icon: '🏷️' },
              { label: 'Meta Tags', score: 80, icon: '📝' },
              { label: 'Content Quality', score: 82, icon: '✍️' },
              { label: 'Technical SEO', score: 85, icon: '⚙️' },
              { label: 'Mobile Ready', score: 90, icon: '📱' },
              { label: 'Performance', score: 72, icon: '⚡' }
            ].map((item, idx) => (
              <div key={idx} style={{ background: colors.neutral100, padding: '18px', borderRadius: '10px' }}>
                <div style={{ display: 'flex', alignItems: 'center', gap: '12px', marginBottom: '10px' }}>
                  <span style={{ fontSize: '2em' }}>{item.icon}</span>
                  <div style={{ flex: 1 }}>
                    <div style={{ display: 'flex', justifyContent: 'space-between', marginBottom: '8px' }}>
                      <span style={{ fontWeight: '700', color: colors.neutral900, fontSize: '1.1em' }}>
                        {item.label}
                      </span>
                      <span style={{
                        fontWeight: '700',
                        fontSize: '1.5em',
                        color: item.score >= 80 ? colors.leaf : item.score >= 60 ? colors.sand : colors.berry
                      }}>
                        {item.score}
                      </span>
                    </div>
                    <div style={{ height: '12px', background: '#E0E0E0', borderRadius: '6px', overflow: 'hidden' }}>
                      <div style={{
                        height: '100%',
                        width: `${item.score}%`,
                        background: item.score >= 80 ? colors.leaf : item.score >= 60 ? colors.sand : colors.berry,
                        transition: 'width 1.5s ease',
                        borderRadius: '6px'
                      }} />
                    </div>
                  </div>
                </div>
              </div>
            ))}
          </div>
        </div>

        {/* Performance Radar */}
        <div style={{
          background: colors.neutral0,
          borderRadius: '12px',
          padding: '30px',
          marginBottom: '25px',
          boxShadow: '0 10px 40px rgba(0,0,0,0.1)',
          border: `3px solid ${colors.sky}`
        }}>
          <h2 style={{
            color: colors.moss,
            marginBottom: '20px',
            fontSize: '1.6em',
            fontWeight: '700',
            borderBottom: `3px solid ${colors.sky}`,
            paddingBottom: '10px'
          }}>
            🎯 Performance Overview
          </h2>
          <ResponsiveContainer width="100%" height={350}>
            <RadarChart data={auditData.radarData}>
              <PolarGrid stroke={colors.neutral500} />
              <PolarAngleAxis dataKey="metric" style={{ fontSize: '13px', fontWeight: '600', fill: colors.neutral900 }} />
              <PolarRadiusAxis angle={90} domain={[0, 100]} stroke={colors.neutral500} />
              <Radar name="Your Site" dataKey="score" stroke={colors.sky} fill={colors.sky} fillOpacity={0.6} strokeWidth={3} />
              <Tooltip contentStyle={{ background: colors.neutral0, border: `3px solid ${colors.sky}`, borderRadius: '8px' }} />
              <Legend />
            </RadarChart>
          </ResponsiveContainer>
        </div>

        {/* Issues */}
        <div style={{
          background: colors.neutral0,
          borderRadius: '12px',
          padding: '30px',
          marginBottom: '25px',
          boxShadow: '0 10px 40px rgba(0,0,0,0.1)',
          border: `3px solid ${colors.sky}`
        }}>
          <h2 style={{
            color: colors.moss,
            marginBottom: '25px',
            fontSize: '1.6em',
            fontWeight: '700',
            borderBottom: `3px solid ${colors.sky}`,
            paddingBottom: '10px'
          }}>
            🎯 Key Issues & Recommendations
          </h2>

          {auditData.issues.map((issue, index) => (
            <div key={index} style={{
              background: colors.neutral100,
              borderLeft: `5px solid ${getSeverityColor(issue.severity)}`,
              padding: '25px',
              marginBottom: '18px',
              borderRadius: '10px',
              position: 'relative'
            }}>
              {issue.priority && (
                <div style={{
                  position: 'absolute',
                  top: '15px',
                  right: '15px',
                  background: colors.sky,
                  color: colors.neutral0,
                  width: '32px',
                  height: '32px',
                  borderRadius: '50%',
                  display: 'flex',
                  alignItems: 'center',
                  justifyContent: 'center',
                  fontWeight: '700'
                }}>
                  {issue.priority}
                </div>
              )}

              <div style={{ display: 'flex', gap: '12px' }}>
                <span style={{ fontSize: '1.8em' }}>{getSeverityIcon(issue.severity)}</span>
                <div style={{ flex: 1 }}>
                  <div style={{ display: 'flex', alignItems: 'center', flexWrap: 'wrap', gap: '10px', marginBottom: '10px' }}>
                    <h4 style={{ margin: 0, color: colors.neutral900, fontSize: '1.2em', fontWeight: '700' }}>
                      {issue.title}
                    </h4>
                    {issue.impact && getImpactBadge(issue.impact)}
                  </div>
                  <p style={{ marginBottom: '12px', color: colors.neutral900, lineHeight: '1.7' }}>
                    {issue.description}
                  </p>
                  {issue.actionable && (
                    <div style={{
                      background: colors.neutral0,
                      padding: '15px',
                      borderRadius: '8px',
                      borderLeft: `3px solid ${colors.petal}`
                    }}>
                      <strong style={{ color: colors.neutral900, display: 'block', marginBottom: '8px' }}>
                        💡 Action Required:
                      </strong>
                      <div style={{ color: colors.neutral900, fontSize: '0.9em' }}>
                        {issue.actionable}
                      </div>
                    </div>
                  )}
                </div>
              </div>
            </div>
          ))}
        </div>

        {/* Schema Comparison */}
        <div style={{
          background: colors.neutral0,
          borderRadius: '12px',
          padding: '30px',
          marginBottom: '25px',
          boxShadow: '0 10px 40px rgba(0,0,0,0.1)',
          border: `3px solid ${colors.sky}`
        }}>
          <h2 style={{
            color: colors.moss,
            marginBottom: '20px',
            fontSize: '1.6em',
            fontWeight: '700',
            borderBottom: `3px solid ${colors.sky}`,
            paddingBottom: '10px'
          }}>
            📈 You vs. Competitors
          </h2>
          <ResponsiveContainer width="100%" height={350}>
            <BarChart data={auditData.schemaComparison}>
              <CartesianGrid strokeDasharray="3 3" stroke={colors.neutral100} />
              <XAxis dataKey="schema" style={{ fontSize: '13px', fontWeight: '600' }} />
              <YAxis />
              <Tooltip contentStyle={{ background: colors.neutral0, border: `3px solid ${colors.sky}`, borderRadius: '8px' }} />
              <Legend />
              <Bar dataKey="yourSite" fill={colors.sky} name="Your Site" radius={[8, 8, 0, 0]} />
              <Bar dataKey="competitors" fill={colors.moss} name="Competitors" radius={[8, 8, 0, 0]} />
            </BarChart>
          </ResponsiveContainer>
        </div>

        {/* Schema Types */}
        <div style={{
          background: colors.neutral0,
          borderRadius: '12px',
          padding: '30px',
          marginBottom: '25px',
          boxShadow: '0 10px 40px rgba(0,0,0,0.1)',
          border: `3px solid ${colors.sky}`
        }}>
          <h2 style={{
            color: colors.moss,
            marginBottom: '20px',
            fontSize: '1.6em',
            fontWeight: '700',
            borderBottom: `3px solid ${colors.sky}`,
            paddingBottom: '10px'
          }}>
            🏷️ Structured Data
          </h2>
          <div style={{ display: 'grid', gridTemplateColumns: 'repeat(auto-fill, minmax(200px, 1fr))', gap: '15px' }}>
            {auditData.schemaTypes.map((schema, idx) => (
              <div key={idx} style={{
                background: schema.status === 'found' ? `${colors.leaf}15` : `${colors.berry}15`,
                padding: '18px',
                borderRadius: '10px',
                borderLeft: `4px solid ${schema.status === 'found' ? colors.leaf : colors.berry}`
              }}>
                <div style={{ display: 'flex', justifyContent: 'space-between', marginBottom: '8px' }}>
                  <div style={{ display: 'flex', alignItems: 'center', gap: '8px' }}>
                    <span style={{ fontSize: '1.4em' }}>{schema.icon}</span>
                    <strong style={{ color: colors.neutral900, fontWeight: '700' }}>{schema.name}</strong>
                  </div>
                  <span style={{ fontSize: '1.5em' }}>{schema.status === 'found' ? '✅' : '❌'}</span>
                </div>
                <div style={{ fontSize: '0.85em', color: colors.neutral900, fontWeight: '600' }}>
                  {schema.status === 'found' ? `Found: ${schema.count}` : 'Missing'}
                </div>
              </div>
            ))}
          </div>
        </div>

        {/* Competitors */}
        <div style={{
          background: colors.neutral0,
          borderRadius: '12px',
          padding: '30px',
          marginBottom: '25px',
          boxShadow: '0 10px 40px rgba(0,0,0,0.1)',
          border: `3px solid ${colors.sky}`
        }}>
          <h2 style={{
            color: colors.moss,
            marginBottom: '20px',
            fontSize: '1.6em',
            fontWeight: '700',
            borderBottom: `3px solid ${colors.sky}`,
            paddingBottom: '10px'
          }}>
            🏆 Top Competitors
          </h2>
          {auditData.competitors.map((comp, index) => (
            <div key={index} style={{
              background: colors.neutral100,
              padding: '20px',
              marginBottom: '12px',
              borderRadius: '10px',
              display: 'flex',
              justifyContent: 'space-between',
              alignItems: 'center'
            }}>
              <div style={{ flex: 1 }}>
                <div style={{ display: 'flex', alignItems: 'center', gap: '12px', marginBottom: '8px' }}>
                  <span style={{
                    background: colors.sky,
                    color: colors.neutral0,
                    width: '35px',
                    height: '35px',
                    borderRadius: '50%',
                    display: 'flex',
                    alignItems: 'center',
                    justifyContent: 'center',
                    fontWeight: '700'
                  }}>
                    #{comp.rank}
                  </span>
                  <div style={{ color: colors.neutral900, fontWeight: '600', fontSize: '1.05em' }}>
                    {comp.url}
                  </div>
                </div>
                <div style={{ display: 'flex', flexWrap: 'wrap', gap: '6px', marginLeft: '47px' }}>
                  {comp.schemas.map((s, i) => (
                    <span key={i} style={{
                      background: colors.neutral0,
                      padding: '4px 10px',
                      borderRadius: '12px',
                      fontSize: '0.8em',
                      color: colors.neutral900,
                      border: `1.5px solid ${colors.neutral500}`,
                      fontWeight: '600'
                    }}>
                      {s}
                    </span>
                  ))}
                </div>
              </div>
              <div style={{
                background: comp.schemaScore >= 90 ? colors.leaf : comp.schemaScore >= 80 ? colors.sand : colors.berry,
                color: colors.neutral0,
                padding: '10px 18px',
                borderRadius: '20px',
                fontWeight: '700',
                fontSize: '1.2em',
                marginLeft: '15px'
              }}>
                {comp.schemaScore}
              </div>
            </div>
          ))}
        </div>

        {/* Opportunities */}
        <div style={{
          background: colors.neutral0,
          borderRadius: '12px',
          padding: '30px',
          boxShadow: '0 10px 40px rgba(0,0,0,0.1)',
          border: `3px solid ${colors.sky}`
        }}>
          <h2 style={{
            color: colors.moss,
            marginBottom: '20px',
            fontSize: '1.6em',
            fontWeight: '700',
            borderBottom: `3px solid ${colors.sky}`,
            paddingBottom: '10px'
          }}>
            💡 Growth Opportunities
          </h2>
          {auditData.opportunities.map((opp, index) => (
            <div key={index} style={{
              background: colors.neutral100,
              padding: '22px',
              marginBottom: '15px',
              borderRadius: '10px',
              borderLeft: `4px solid ${colors.petal}`
            }}>
              <div style={{ display: 'flex', justifyContent: 'space-between', marginBottom: '10px', flexWrap: 'wrap', gap: '10px' }}>
                <h3 style={{ color: colors.neutral900, fontSize: '1.15em', fontWeight: '700', margin: 0 }}>
                  {index + 1}. {opp.title}
                </h3>
                <div style={{ display: 'flex', gap: '8px' }}>
                  {getImpactBadge(opp.impact)}
                  {getEffortBadge(opp.effort)}
                </div>
              </div>
              <p style={{ color: colors.neutral900, marginBottom: '12px', lineHeight: '1.6' }}>
                {opp.description}
              </p>
              <div style={{ background: colors.neutral0, padding: '12px', borderRadius: '8px' }}>
                <strong style={{ color: colors.neutral900, fontSize: '0.95em' }}>Steps:</strong>
                <ol style={{ marginLeft: '20px', marginTop: '8px', color: colors.neutral900, lineHeight: '1.7', fontSize: '0.9em' }}>
                  {opp.steps.map((step, i) => <li key={i}>{step}</li>)}
                </ol>
              </div>
            </div>
          ))}
        </div>

        {/* Footer */}
        <div style={{ textAlign: 'center', color: colors.neutral0, padding: '30px', marginTop: '30px' }}>
          <WordLiftLogo />
          <p style={{ fontSize: '0.9em', opacity: 0.9, marginTop: '10px' }}>
            Enterprise SEO Solutions
          </p>
        </div>
      </div>
    </div>
  );
};

export default WordLiftSEODashboard;
```

### Usage Instructions

**Dependencies Required:**
```bash
npm install react recharts
```

**Data Structure:**
The component expects an `auditData` object matching the structure returned by WordLift MCP tools. Key properties:

- `score` (number): Overall SEO health score 0-100
- `scores` (object): Individual dimension scores
- `issues` (array): List of issues with severity, category, title, description, impact, actionable steps
- `schemaTypes` (array): Schema types found/missing with icons
- `competitors` (array): Competitor data with schema scores
- `opportunities` (array): Growth opportunities with impact/effort ratings
- `radarData` (array): Six-dimensional performance data
- `schemaComparison` (array): Your site vs competitors comparison

**Customization:**
- Update `colors` object to match your brand
- Modify chart colors in Recharts components
- Adjust responsive breakpoints in grid layouts
- Customize logo and branding elements

**Integration:**
This component can be embedded in React applications or exported as a standalone HTML report using tools like React-to-HTML or Next.js static generation.
